#!/usr/bin/env python3
import os
import shutil
import subprocess
import zipfile
import requests
from datetime import datetime
from rgbmatrix import RGBMatrix, RGBMatrixOptions, graphics

##############################
# Path Constants
##############################

BASE_PATHS = {
    "core_dir": "/home/admin/Core",
    "backup_dir": "/home/admin/Core-Backup",
    "tracker_dir": "/home/admin/plane-tracker-rgb-pi",
    "font_path": "/home/admin/rpi-rgb-led-matrix/fonts/5x7.bdf",
    "base_url": "https://planesintheclouds.com/pitc-core/core-deployment/",
    "zip_filename": "its-a-plane-python.zip",
    "config_url": "https://planesintheclouds.com/pitc-core/core-deployment/config.py"
}
BASE_PATHS["zip_url"] = BASE_PATHS["base_url"] + BASE_PATHS["zip_filename"]
BASE_PATHS["zip_path"] = f"{BASE_PATHS['core_dir']}/its-a-plane-python.zip"
BASE_PATHS["extracted_path"] = f"{BASE_PATHS['core_dir']}/its-a-plane-python"
BASE_PATHS["tracker_extracted"] = f"{BASE_PATHS['tracker_dir']}/its-a-plane-python"
BASE_PATHS["config_path"] = f"{BASE_PATHS['tracker_extracted']}/config.py"

##############################
# Display Functions (Display.py)
##############################

def setup_matrix():
    options = RGBMatrixOptions()
    options.rows = 32
    options.cols = 64
    options.chain_length = 1
    options.parallel = 1
    options.hardware_mapping = 'adafruit-hat'
    options.gpio_slowdown = 2
    options.brightness = 75
    return RGBMatrix(options=options)

def draw_progress(matrix, canvas, percent, font, color_text, color_bar):
    canvas.Clear()
    init_text = "INITIALIZING"
    text_width = graphics.DrawText(canvas, font, 0, 0, color_text, init_text)
    text_x = (64 - text_width) // 2
    graphics.DrawText(canvas, font, text_x, 13, color_text, init_text)

    bar_x = 6
    bar_y = 14
    bar_width = 52
    bar_height = 8

    graphics.DrawLine(canvas, bar_x, bar_y, bar_x + bar_width, bar_y, color_text)
    graphics.DrawLine(canvas, bar_x, bar_y, bar_x, bar_y + bar_height, color_text)
    graphics.DrawLine(canvas, bar_x + bar_width, bar_y, bar_x + bar_width, bar_y + bar_height, color_text)
    graphics.DrawLine(canvas, bar_x, bar_y + bar_height, bar_x + bar_width, bar_y + bar_height, color_text)

    fill_width = int((percent / 100.0) * (bar_width - 1))
    for x in range(bar_x + 1, bar_x + 1 + fill_width):
        for y in range(bar_y + 1, bar_y + bar_height):
            canvas.SetPixel(x, y, color_bar.red, color_bar.green, color_bar.blue)

    percent_text = f"{int(percent)}%"
    percent_width = graphics.DrawText(canvas, font, 0, 0, color_text, percent_text)
    percent_x = (64 - percent_width) // 2
    graphics.DrawText(canvas, font, percent_x, 30, color_text, percent_text)

    canvas = matrix.SwapOnVSync(canvas)
    return canvas

##############################
# Utility Functions (Utils.py)
##############################

def ensure_directory(path):
    try:
        os.makedirs(path, exist_ok=True)
        subprocess.run(["chmod", "-R", "777", path], check=True)
        print(f"Ensured directory exists with permissions: {path}")
    except Exception as e:
        print(f"Failed to ensure directory {path}: {e}")

def set_permissions():
    commands = [
        f"sudo chown -R admin:admin {BASE_PATHS['tracker_dir']}/", 
        f"sudo chown -R admin:admin {BASE_PATHS['core_dir']}/", 
        f"sudo chown -R admin:admin {BASE_PATHS['backup_dir']}/",
        f"sudo chmod -R u+rwX {BASE_PATHS['core_dir']}/", 
        f"sudo chmod -R u+rwX {BASE_PATHS['backup_dir']}/",
        f"sudo chmod -R u+rwX {BASE_PATHS['tracker_dir']}/"
    ]
    for cmd in commands:
        try:
            subprocess.run(cmd, shell=True, check=True)
        except subprocess.CalledProcessError as e:
            print(f"Error setting permissions with command {cmd}: {e}")

def clean_core_directory(path=BASE_PATHS['core_dir']):
    if os.path.exists(path):
        for item in os.listdir(path):
            item_path = os.path.join(path, item)
            try:
                if os.path.isfile(item_path) or os.path.islink(item_path):
                    os.remove(item_path)
                elif os.path.isdir(item_path):
                    shutil.rmtree(item_path)
            except Exception as e:
                print(f"Error removing {item_path}: {e}")
    else:
        os.makedirs(path, exist_ok=True)

def download_zip_file(url, save_path, matrix, canvas, font, color_text, color_bar):
    try:
        with requests.get(url, stream=True, timeout=10) as response:
            response.raise_for_status()
            total_size = int(response.headers.get('content-length', 0))
            downloaded = 0
            with open(save_path, 'wb') as f:
                for chunk in response.iter_content(chunk_size=8192):
                    if chunk:
                        f.write(chunk)
                        downloaded += len(chunk)
                        percent = (downloaded / total_size) * 100 if total_size else 0
                        canvas = draw_progress(matrix, canvas, percent, font, color_text, color_bar)
        print(f"Downloaded to: {save_path}")
    except Exception as e:
        print(f"Failed to download zip file: {e}")

def extract_zip_file(zip_path, extract_to):
    try:
        with zipfile.ZipFile(zip_path, 'r') as zip_ref:
            zip_ref.extractall(extract_to)
        print(f"Extracted {zip_path} to {extract_to}")
    except Exception as e:
        print(f"Failed to extract zip file: {e}")

def backup_zip_file(source_path, backup_dir=BASE_PATHS['backup_dir'], max_backups=3):
    try:
        os.makedirs(backup_dir, exist_ok=True)
        timestamp = datetime.now().strftime("%Y-%m-%d_%H-%M-%S")
        filename = f"its-a-plane-python_{timestamp}.zip"
        backup_path = os.path.join(backup_dir, filename)
        shutil.copy2(source_path, backup_path)
        print(f"Backup created at: {backup_path}")

        backups = sorted(
            [f for f in os.listdir(backup_dir) if f.endswith(".zip")],
            key=lambda x: os.path.getmtime(os.path.join(backup_dir, x)),
            reverse=True
        )

        for old_file in backups[max_backups:]:
            old_path = os.path.join(backup_dir, old_file)
            os.remove(old_path)
            print(f"Deleted old backup: {old_path}")

    except Exception as e:
        print(f"Failed to manage backups: {e}")

def set_full_access_permissions(path=BASE_PATHS['extracted_path']):
    try:
        subprocess.run(["chmod", "-R", "777", path], check=True)
        print(f"Full access permissions set for: {path}")
    except subprocess.CalledProcessError as e:
        print(f"Failed to set full permissions for {path}: {e}")

def copy_to_tracker_dir(source=BASE_PATHS['extracted_path'], destination=BASE_PATHS['tracker_dir']):
    try:
        target_path = os.path.join(destination, os.path.basename(source))
        if os.path.exists(target_path):
            shutil.rmtree(target_path)
        shutil.copytree(source, target_path)
        print(f"Copied {source} to {target_path}")
    except Exception as e:
        print(f"Failed to copy to tracker directory: {e}")

def download_config():
    destination_file = BASE_PATHS['config_path']
    os.makedirs(os.path.dirname(destination_file), exist_ok=True)
    try:
        response = requests.get(BASE_PATHS['config_url'], timeout=10)
        response.raise_for_status()
        with open(destination_file, "wb") as f:
            f.write(response.content)
        print(f"Downloaded config.py to {destination_file}")
    except Exception as e:
        print(f"Error downloading config.py: {e}")

##############################
# Main Execution (Main.py)
##############################

if __name__ == "__main__":
    ensure_directory(BASE_PATHS['core_dir'])
    ensure_directory(BASE_PATHS['backup_dir'])
    ensure_directory(BASE_PATHS['tracker_dir'])

    set_permissions()
    clean_core_directory()

    matrix = setup_matrix()
    font = graphics.Font()
    if not os.path.isfile(BASE_PATHS['font_path']):
        raise FileNotFoundError(f"Font not found: {BASE_PATHS['font_path']}")
    font.LoadFont(BASE_PATHS['font_path'])

    text_white = graphics.Color(255, 255, 255)
    progress_green = graphics.Color(0, 255, 0)

    canvas = matrix.CreateFrameCanvas()
    canvas = draw_progress(matrix, canvas, 0, font, text_white, progress_green)

    download_zip_file(BASE_PATHS['zip_url'], BASE_PATHS['zip_path'], matrix, canvas, font, text_white, progress_green)
    backup_zip_file(BASE_PATHS['zip_path'])
    extract_zip_file(BASE_PATHS['zip_path'], BASE_PATHS['core_dir'])
    set_full_access_permissions(BASE_PATHS['extracted_path'])
    copy_to_tracker_dir()
    download_config()
    clean_core_directory()
    canvas = draw_progress(matrix, canvas, 100, font, text_white, progress_green)
