from FlightRadar24.api import FlightRadar24API
from threading import Thread, Lock
from time import sleep
import math
from typing import Optional, Tuple
from config import DISTANCE_UNITS
from datetime import datetime

from requests.exceptions import ConnectionError
from urllib3.exceptions import NewConnectionError, MaxRetryError

try:
    from config import MIN_ALTITUDE, MAX_ALTITUDE
except (ModuleNotFoundError, NameError, ImportError):
    MIN_ALTITUDE = 0
    MAX_ALTITUDE = 100000

try:
    from config import RETRIES
except (ModuleNotFoundError, NameError, ImportError):
    RETRIES = 3

try:
    from config import RATE_LIMIT_DELAY
except (ModuleNotFoundError, NameError, ImportError):
    RATE_LIMIT_DELAY = 5

try:
    from config import MAX_FLIGHT_LOOKUP
except (ModuleNotFoundError, NameError, ImportError):
    MAX_FLIGHT_LOOKUP = 5

EARTH_RADIUS_M = 3958.8
BLANK_FIELDS = ["", "N/A", "NONE"]

try:
    from config import ZONE_HOME, LOCATION_HOME
    ZONE_DEFAULT = ZONE_HOME
    LOCATION_DEFAULT = LOCATION_HOME
except (ModuleNotFoundError, NameError, ImportError):
    ZONE_DEFAULT = {"tl_y": 62.61, "tl_x": -13.07, "br_y": 49.71, "br_x": 3.46}
    LOCATION_DEFAULT = [51.509865, -0.118092, EARTH_RADIUS_M]

try:
    from config import QUIET_START, QUIET_END
except (ModuleNotFoundError, NameError, ImportError):
    QUIET_START = None
    QUIET_END = None

quiet_state = {"in_quiet_time": False}

def parse_time(timestr):
    return datetime.strptime(timestr, "%H:%M").time()

def is_quiet_time():
    if not QUIET_START or not QUIET_END:
        return False

    now = datetime.now().time()
    start = parse_time(QUIET_START)
    end = parse_time(QUIET_END)

    if start < end:
        in_quiet = start <= now < end
    else:
        in_quiet = now >= start or now < end

    previously_quiet = quiet_state.get("in_quiet_time", False)

    if in_quiet and not previously_quiet:
        print("Quiet time started.")
        print("Currently in quiet time. Skipping data fetch.")
    elif not in_quiet and previously_quiet:
        print("Quiet time ended.")

    quiet_state["in_quiet_time"] = in_quiet
    return in_quiet

def polar_to_cartesian(lat, long, alt):
    DEG2RAD = math.pi / 180
    return [
        alt * math.cos(DEG2RAD * lat) * math.sin(DEG2RAD * long),
        alt * math.sin(DEG2RAD * lat),
        alt * math.cos(DEG2RAD * lat) * math.cos(DEG2RAD * long),
    ]

def distance_from_flight_to_home(flight, home=LOCATION_DEFAULT):
    try:
        lat1, lon1 = math.radians(flight.latitude), math.radians(flight.longitude)
        lat2, lon2 = math.radians(home[0]), math.radians(home[1])
        dlat = lat2 - lat1
        dlon = lon2 - lon1
        a = math.sin(dlat / 2)**2 + math.cos(lat1) * math.cos(lat2) * math.sin(dlon / 2)**2
        c = 2 * math.atan2(math.sqrt(a), math.sqrt(1 - a))
        dist_miles = EARTH_RADIUS_M * c
        return dist_miles * 1.609 if DISTANCE_UNITS == "metric" else dist_miles
    except AttributeError:
        return 1e6

def plane_bearing(flight, home=LOCATION_DEFAULT):
    lat1 = math.radians(home[0])
    long1 = math.radians(home[1])
    lat2 = math.radians(flight.latitude)
    long2 = math.radians(flight.longitude)
    bearing = math.atan2(
        math.sin(long2 - long1) * math.cos(lat2),
        math.cos(lat1) * math.sin(lat2) - math.sin(lat1) * math.cos(lat2) * math.cos(long2 - long1)
    )
    bearing = math.degrees(bearing)
    return (bearing + 360) % 360

def degrees_to_cardinal(d):
    dirs = ["N", "NE", "E", "SE", "S", "SW", "W", "NW"]
    ix = int((d + 22.5)/45)
    return dirs[ix % 8]

def distance_from_flight_to_origin(flight, origin_latitude, origin_longitude, origin_altitude):
    try:
        return distance_between(flight.latitude, flight.longitude, origin_latitude, origin_longitude)
    except Exception as e:
        print("Error:", e)
        return None

def distance_from_flight_to_destination(flight, destination_latitude, destination_longitude, destination_altitude):
    try:
        return distance_between(flight.latitude, flight.longitude, destination_latitude, destination_longitude)
    except Exception as e:
        print("Error:", e)
        return None

def distance_between(lat1, lon1, lat2, lon2):
    dlat = math.radians(lat2 - lat1)
    dlon = math.radians(lon2 - lon1)
    a = math.sin(dlat / 2)**2 + math.cos(math.radians(lat1)) * math.cos(math.radians(lat2)) * math.sin(dlon / 2)**2
    c = 2 * math.atan2(math.sqrt(a), math.sqrt(1 - a))
    dist = EARTH_RADIUS_M * c
    return dist * 1.609 if DISTANCE_UNITS == "metric" else dist

class Overhead:
    def __init__(self):
        self._api = FlightRadar24API()
        self._lock = Lock()
        self._data = []
        self._new_data = False
        self._processing = False

    def grab_data(self):
        Thread(target=self._grab_data).start()

    def _grab_data(self):
        if is_quiet_time():
            with self._lock:
                self._new_data = False
                self._processing = False
            return

        with self._lock:
            self._new_data = False
            self._processing = True

        data = []

        try:
            bounds = self._api.get_bounds(ZONE_DEFAULT)
            flights = self._api.get_flights(bounds=bounds)
            flights = [
                f for f in flights
                if f.altitude < MAX_ALTITUDE and f.altitude > MIN_ALTITUDE
            ]
            flights = sorted(flights, key=lambda f: distance_from_flight_to_home(f))

            for flight in flights[:MAX_FLIGHT_LOOKUP]:
                if is_quiet_time():
                    print("Quiet time detected mid-processing. Aborting fetch.")
                    data = []
                    break

                sleep(RATE_LIMIT_DELAY)

                retries = RETRIES
                while retries:
                    try:
                        details = self._api.get_flight_details(flight)

                        try:
                            plane = details["aircraft"]["model"]["code"]
                        except (KeyError, TypeError):
                            plane = ""
                        plane = plane if not (plane.upper() in BLANK_FIELDS) else ""

                        try:
                            registration = details["aircraft"]["registration"]
                        except (KeyError, TypeError):
                            registration = ""

                        origin = flight.origin_airport_iata if not (flight.origin_airport_iata.upper() in BLANK_FIELDS) else ""
                        destination = flight.destination_airport_iata if not (flight.destination_airport_iata.upper() in BLANK_FIELDS) else ""
                        callsign = flight.callsign if not (flight.callsign.upper() in BLANK_FIELDS) else ""

                        try:
                            airline = details["airline"]["name"]
                        except (KeyError, TypeError):
                            airline = ""

                        try:
                            time_scheduled_departure = details["time"]["scheduled"]["departure"]
                            time_scheduled_arrival = details["time"]["scheduled"]["arrival"]
                            time_real_departure = details["time"]["real"]["departure"]
                            time_estimated_arrival = details["time"]["estimated"]["arrival"]
                        except (KeyError, TypeError):
                            time_scheduled_departure = time_scheduled_arrival = None
                            time_real_departure = time_estimated_arrival = None

                        origin_latitude = origin_longitude = origin_altitude = None
                        if details['airport']['origin']:
                            origin_latitude = details['airport']['origin']['position']['latitude']
                            origin_longitude = details['airport']['origin']['position']['longitude']
                            origin_altitude = details['airport']['origin']['position']['altitude']

                        destination_latitude = destination_longitude = destination_altitude = None
                        if details['airport']['destination']:
                            destination_latitude = details['airport']['destination']['position']['latitude']
                            destination_longitude = details['airport']['destination']['position']['longitude']
                            destination_altitude = details['airport']['destination']['position']['altitude']

                        distance_origin = distance_from_flight_to_origin(flight, origin_latitude, origin_longitude, origin_altitude) if origin_latitude else 0
                        distance_destination = distance_from_flight_to_destination(flight, destination_latitude, destination_longitude, destination_altitude) if destination_latitude else 0

                        try:
                            owner_icao = details["owner"]["code"]["icao"]
                        except (KeyError, TypeError):
                            owner_icao = flight.airline_icao if not (flight.airline_icao.upper() in BLANK_FIELDS) else ""
                        owner_iata = flight.airline_iata or "N/A"

                        data.append({
                            "airline": airline,
                            "plane": plane,
                            "origin": origin,
                            "owner_iata": owner_iata,
                            "owner_icao": owner_icao,
                            "destination": destination,
                            "registration": registration,
                            "time_scheduled_departure": time_scheduled_departure,
                            "time_scheduled_arrival": time_scheduled_arrival,
                            "time_real_departure": time_real_departure,
                            "time_estimated_arrival": time_estimated_arrival,
                            "vertical_speed": flight.vertical_speed,
                            "altitude": flight.altitude,
                            "ground_speed": flight.ground_speed if hasattr(flight, 'ground_speed') else None,
                            "callsign": callsign,
                            "distance_origin": distance_origin,
                            "distance_destination": distance_destination,
                            "distance": distance_from_flight_to_home(flight),
                            "direction": degrees_to_cardinal(plane_bearing(flight)),
                        })
                        break
                    except (KeyError, AttributeError):
                        retries -= 1
                        sleep(RATE_LIMIT_DELAY * (RETRIES - retries + 1))

            with self._lock:
                self._new_data = True
                self._processing = False
                self._data = data

        except (ConnectionError, NewConnectionError, MaxRetryError):
            with self._lock:
                self._new_data = False
                self._processing = False

    @property
    def new_data(self):
        with self._lock:
            return self._new_data

    @property
    def processing(self):
        with self._lock:
            return self._processing

    @property
    def data(self):
        with self._lock:
            self._new_data = False
            return self._data

    @property
    def data_is_empty(self):
        with self._lock:
            return len(self._data) == 0

if __name__ == "__main__":
    o = Overhead()
    o.grab_data()
    while not o.new_data:
        print("processing...")
        sleep(1)
    print(o.data)
