from time import sleep
import inspect

DELAY_DEFAULT = 0.01

class Animator(object):
    class KeyFrame(object):
        @staticmethod
        def add(divisor, offset=0):
            def wrapper(func):
                func.properties = {"divisor": divisor, "offset": offset, "count": 0}
                return func
            return wrapper

    def __init__(self):
        self.keyframes = []
        self.frame = 0
        self._delay = DELAY_DEFAULT
        self._reset_scene = True
        self._register_keyframes()
        super().__init__()

    def _register_keyframes(self):
        for methodname in dir(self):
            method = getattr(self, methodname)
            if hasattr(method, "properties"):
                self.keyframes.append(method)

    def _safe_call(self, method, arg):
        """Call method with arg only if it expects a parameter in addition to 'self'."""
        argspec = inspect.getfullargspec(method)
        if len(argspec.args) >= 2:
            return method(arg)
        else:
            return method()

    def reset_scene(self):
        for keyframe in self.keyframes:
            if keyframe.properties["divisor"] == 0:
                self._safe_call(keyframe, 0)

    def play(self):
        while True:
            for keyframe in self.keyframes:
                if self.frame == 0:
                    if keyframe.properties["divisor"] == 0:
                        self._safe_call(keyframe, 0)

                if (
                    self.frame > 0
                    and keyframe.properties["divisor"]
                    and not (
                        (self.frame - keyframe.properties["offset"])
                        % keyframe.properties["divisor"]
                    )
                ):
                    result = self._safe_call(keyframe, keyframe.properties["count"])
                    if result is True:
                        keyframe.properties["count"] = 0
                    else:
                        keyframe.properties["count"] += 1

            self._reset_scene = False
            self.frame += 1
            sleep(self._delay)

    @property
    def delay(self):
        return self._delay

    @delay.setter
    def delay(self, value):
        self._delay = value
