import math
from rgbmatrix import graphics
from utilities.animator import Animator
from setup import colours, fonts, screen
from config import DISTANCE_UNITS

# Setup
PLANE_COLOUR = colours.WHITE
PLANE_DISTANCE_COLOUR = colours.WHITE
PLANE_DISTANCE_FROM_TOP = 31
PLANE_TEXT_HEIGHT = 6
PLANE_FONT = fonts.small

# Transition altitude mapping based on region
TRANSITION_ALTITUDES = {
    "Europe": 5000,
    "US": 18000,
    "Australia": 10000,
    "Default": 18000
}

def get_timezone():
    try:
        with open("/etc/timezone") as f:
            return f.read().strip()
    except Exception:
        return "Default"

def get_transition_altitude(timezone):
    if "Europe" in timezone:
        return TRANSITION_ALTITUDES["Europe"]
    elif "US" in timezone or "America" in timezone:
        return TRANSITION_ALTITUDES["US"]
    elif "Australia" in timezone:
        return TRANSITION_ALTITUDES["Australia"]
    else:
        return TRANSITION_ALTITUDES["Default"]

class PlaneDetailsScene(object):
    def __init__(self):
        super().__init__()
        self.plane_position = screen.WIDTH
        self._data = []
        self._data_index = 0
        self._data_all_looped = False
        self.timezone = get_timezone()
        self.transition_altitude = get_transition_altitude(self.timezone)

    @Animator.KeyFrame.add(1)
    def plane_details(self, count):
        if not self._data:
            return

        # Extract data
        plane_data = self._data[self._data_index]
        plane_name = plane_data["plane"]
        registration = plane_data.get("registration", "")
        distance = plane_data["distance"]
        direction = plane_data["direction"]
        altitude = plane_data.get("altitude")
        ground_speed = plane_data.get("ground_speed")
        distance_units = "mi" if DISTANCE_UNITS == "imperial" else "KM"

        # Text construction
        plane_name_text = f'{plane_name} Reg:{registration} '
        distance_text = f'Location: {math.ceil(distance)}{distance_units} {direction} '

        alt_speed_parts = []
        if isinstance(altitude, (int, float)):
            if altitude > self.transition_altitude:
                alt_speed_parts.append(f'FL{int(altitude // 100):03}')
            else:
                alt_speed_parts.append(f'Alt:{altitude}ft')
        if isinstance(ground_speed, (int, float)):
            alt_speed_parts.append(f'Gnd Spd:{ground_speed}kts')
        alt_speed_text = ' '.join(alt_speed_parts)

        # Draw background
        self.draw_square(
            0,
            PLANE_DISTANCE_FROM_TOP - PLANE_TEXT_HEIGHT,
            screen.WIDTH,
            screen.HEIGHT,
            colours.BLACK,
        )

        # Draw text with proper spacing
        x = self.plane_position
        plane_name_width = graphics.DrawText(
            self.canvas,
            PLANE_FONT,
            x,
            PLANE_DISTANCE_FROM_TOP,
            PLANE_COLOUR,
            plane_name_text,
        )

        x += plane_name_width
        distance_text_width = graphics.DrawText(
            self.canvas,
            PLANE_FONT,
            x,
            PLANE_DISTANCE_FROM_TOP,
            PLANE_DISTANCE_COLOUR,
            distance_text,
        )

        x += distance_text_width
        alt_speed_text_width = graphics.DrawText(
            self.canvas,
            PLANE_FONT,
            x,
            PLANE_DISTANCE_FROM_TOP,
            PLANE_DISTANCE_COLOUR,
            alt_speed_text,
        ) if alt_speed_text else 0

        # Handle scrolling
        total_text_width = plane_name_width + distance_text_width + alt_speed_text_width
        self.plane_position -= 1

        if self.plane_position + total_text_width < 0:
            self.plane_position = screen.WIDTH
            if len(self._data) > 1:
                self._data_index = (self._data_index + 1) % len(self._data)
                self._data_all_looped |= self._data_index == 0
                self.reset_scene()

    @Animator.KeyFrame.add(0)
    def reset_scrolling(self):
        self.plane_position = screen.WIDTH
