from datetime import datetime, timedelta
from PIL import Image
import os

from utilities.animator import Animator
from setup import colours, fonts, frames, screen
from utilities.temperature import grab_forecast
from config import NIGHT_START, NIGHT_END
from rgbmatrix import graphics

# Pillow version compatibility: use Resampling.LANCZOS or fallback to LANCZOS
try:
    RESAMPLING = Image.Resampling.LANCZOS
except AttributeError:
    RESAMPLING = Image.LANCZOS

DAY_COLOUR = colours.LIGHT_PINK
MIN_T_COLOUR = colours.LIGHT_MID_BLUE
MAX_T_COLOUR = colours.LIGHT_DARK_ORANGE
TEXT_FONT = fonts.extrasmall
FONT_HEIGHT = 5
DISTANCE_FROM_TOP = 32
ICON_SIZE = 10
FORECAST_SIZE = FONT_HEIGHT * 2 + ICON_SIZE
DAY_POSITION = DISTANCE_FROM_TOP - FONT_HEIGHT - ICON_SIZE
ICON_POSITION = DISTANCE_FROM_TOP - FONT_HEIGHT - ICON_SIZE
TEMP_POSITION = DISTANCE_FROM_TOP
NIGHT_START_TIME = datetime.strptime(NIGHT_START, "%H:%M")
NIGHT_END_TIME = datetime.strptime(NIGHT_END, "%H:%M")

class DaysForecastScene(object):
    def __init__(self):
        super().__init__()
        self._redraw_forecast = True
        self._last_hour = None
        self._cached_forecast = None

    @Animator.KeyFrame.add(frames.PER_SECOND * 1)
    def day(self, count):
        now = datetime.now().replace(microsecond=0).time()
        if now == NIGHT_START_TIME.time() or now == NIGHT_END_TIME.time():
            self._redraw_forecast = True
            return False    

        if len(self._data):
            self._redraw_forecast = True
            return False

        current_hour = datetime.now().hour

        if self._last_hour != current_hour or self._redraw_forecast:
            if self._last_hour is not None:
                self.draw_square(
                    0,
                    12,
                    64,
                    32,
                    colours.BLACK,
                )
            self._last_hour = current_hour

            if self._cached_forecast is not None and self._redraw_forecast:
                forecast = self._cached_forecast
            else:
                forecast = grab_forecast()
                self._cached_forecast = forecast

            if forecast is not None:
                self._redraw_forecast = False
                offset = 1
                space_width = screen.WIDTH // 3

                for day in forecast:
                    day_name = datetime.fromisoformat(day["startTime"].rstrip("Z")).strftime("%a")
                    icon = day["values"]["weatherCodeFullDay"]

                    min_temp = f"{day['values']['temperatureMin']:.0f}"
                    max_temp = f"{day['values']['temperatureMax']:.0f}"
                    
                    min_temp_width = len(min_temp) * 4
                    max_temp_width = len(max_temp) * 4

                    temp_x = offset + (space_width - min_temp_width - max_temp_width - 1) // 2 + 1
                    min_temp_x = temp_x + max_temp_width
                    max_temp_x = temp_x
                    icon_x = offset + (space_width - ICON_SIZE) // 2
                    day_x = offset + (space_width - 12) // 2 + 1

                    _ = graphics.DrawText(
                        self.canvas,
                        TEXT_FONT,
                        day_x,
                        DAY_POSITION,
                        DAY_COLOUR,
                        day_name
                    )

                    icons_dir = "/home/admin/plane-tracker-rgb-pi/its-a-plane-python/icons"
                    logos_dir = "/home/admin/plane-tracker-rgb-pi/its-a-plane-python/logos"
                    icon_path = f"{icons_dir}/{icon}.png"
                    fallback_path = f"{icons_dir}/default.png"
                    alt_icon_path = f"{logos_dir}/{icon}.png"

                    if os.path.exists(icon_path):
                        image = Image.open(icon_path)
                    elif os.path.exists(alt_icon_path):
                        print(f"[Info] Using alternate icon from logos: {alt_icon_path}")
                        image = Image.open(alt_icon_path)
                    else:
                        print(f"[Warning] Icon not found: {icon_path} or {alt_icon_path}, using fallback.")
                        image = Image.open(fallback_path)

                    image.thumbnail((ICON_SIZE, ICON_SIZE), RESAMPLING)
                    self.matrix.SetImage(image.convert('RGB'), icon_x, ICON_POSITION)

                    self.draw_square(
                        min_temp_x,
                        TEMP_POSITION - FONT_HEIGHT,
                        max_temp_x + max_temp_width,
                        TEMP_POSITION + FONT_HEIGHT,
                        colours.BLUE
                    )

                    _ = graphics.DrawText(
                        self.canvas,
                        TEXT_FONT,
                        min_temp_x,
                        TEMP_POSITION,
                        MIN_T_COLOUR,
                        min_temp
                    )

                    _ = graphics.DrawText(
                        self.canvas,
                        TEXT_FONT,
                        max_temp_x,
                        TEMP_POSITION,
                        MAX_T_COLOUR,
                        max_temp
                    )

                    offset += space_width

        return False
